<?php
use Joomla\Registry\Registry;
/**
 * @package		com_contactenhanced
 * @author		Douglas Machado {@link http://idealextensions.com} based on the work of Heiko Holtkamp <heiko@rvs.uni-bielefeld.de>
 * @author		Created on 28-Jul-09
 * @license		GNU/GPL, see license.txt
 */

/*
 * include class for the geocoder
 */
if(!class_exists('simpleGMapGeocoder')){
	require_once ('simpleGMapGeocoder.php');
}
class CEGMaps  extends JObject{
	/**
	 * @param string MapCanvas
	 * div id for the map container
	 */
	protected $mapCanvas = 'gmaps_canvas';

	protected $editMode			= false;
	protected $useDirections	= true;
	protected $showCoordinates	= true;
	protected $language			= 'auto';
	protected $mapStyles		= '[]';
	
	

	/**
	 * mapMarkers : array
	 * Holds data (coords etc.) of Markers
	 */
	protected $mapMarkers = array ();


	/**
	 * mapCircles : array
	 * Holds data (coords etc.) of Circles
	 */
	protected $mapCircles = array ();

	/**
	 * mapRectangles : array
	 * Holds data (coords etc.) of Rectangles
	 */
	private $mapRectangles = array ();

	/**
	 * mapHeight
	 * height of the Google Map, in pixels
	 */
	protected $mapHeight = 0;

	/**
	 * apiSensor : boolean
	 * True/False wether the device has a sensor or not
	 */
	protected $apiSensor = true;

	/**
	 * mapDraggable : boolean
	 * True/False wether the map is draggable or not
	 */
	protected $mapDraggable = 1;

	/**
	 * mapType
	 * string : can be 'ROADMAP', 'SATELLITE', 'HYBRID' or 'TERRAIN'
	 * display's either a (road)map, satellite, hybrid or terrain view, (road)map by default
	 */
	protected $mapType = 'ROADMAP';

	/**
	 * zoomLevel
	 * int : 0 - 19
	 * set's the initial zoom level of the map (0 is fully zoomed out and 19 is fully zoomed in)
	 */
	protected $zoomLevel = 10;

	/**
	 * showDefaultUI
	 * True/False whether to show the deault UI controls or not
	 */
	protected $showDefaultUI = true;

	/**
	 * showMapTypeControl
	 * True/False whether to show map type control or not
	 */
	protected $mapTypeControl = 1;

	/**
	 * mapTypeControlStyle
	 * string : can be 'HORIZONTAL_BAR', 'DROPDOWN_MENU' or 'DEFAULT'
	 * Style of the map type control
	 */
	protected $mapTypeControlStyle = 'DEFAULT';

	/**
	 * showNavigationControl
	 * True/False whether to show navigation control or not
	 */
	protected $navigationControl = 1;

	/**
	 * @var string can be 'DRIVING', 'BICYCLING', 'WALKING' or 'TRANSIT'
	 */
	protected $travelMode = 'DRIVING';

	/**
	 * navigationControlStyle
	 * string : can be 'ANDROID', 'DEFAULT', 'SMALL' or 'ZOOM_PAN'
	 * Style of the navigation control
	 */
	protected $navigationControlStyle = 'DEFAULT';

	/**
	 * showScaleControl
	 * True/False whether to show scale control or not
	 */
	protected $scaleControl = 1;

	/**
	 * showStreetViewControl
	 * True/False whether to show StreetView control or not
	 */
	protected $streetViewControl = 1;

	/**
	 * scrollwheelZoom
	 * True/False whether the scrollwhell zoom is enabled or not
	 */
	protected $scrollwheelZoom 		= 1;

	/**
	 * enableDoubleClickZoom
	 * True/False whether doubleclick zoom zoom is enabled or not
	 */
	protected $enableDoubleClickZoom = 1;

	/**
	 * infoWindowBehaviour
	 * string : can be 'MULTIPLE', 'SINGLE', 'CLOSE_ON_MAPCLICK' or 'SINGLE_CLOSE_ON_MAPCLICK'
	 * Behavious of InfoWindow overlays
	 */
	protected $infoWindowBehaviour = 'MULTIPLE';

	/**
	 * infoWindowTrigger
	 * string : can be 'click' OR 'mouseover'
	 * Determines if InfoWindow is shown with a click or by mouseover
	 */
	protected $infoWindowTrigger = 'mouseover';

	/**
	 * maximum longitude of all markers
	 *
	 * @var float
	 */
	private $maxLng = - 1000000;

	/**
	 * minimum longitude of all markers
	 *
	 * @var float
	 */
	private $minLng = 1000000;

	/**
	 * max latitude
	 *
	 * @var float
	 */
	private $maxLat = - 1000000;

	/**
	 * min latitude
	 *
	 * @var float
	 */
	private $minLat = 1000000;

	/**
	 * map center latitude (horizontal)
	 * calculated automatically as markers
	 * are added to the map.
	 *
	 * @var float
	 */
	private $centerLat = null;

	/**
	 * map center longitude (vertical)
	 * calculated automatically as markers
	 * are added to the map.
	 *
	 * @var float
	 */
	private $centerLng = null;

	/**
	 * @function __construct
	 * ription constructor
	 *
	 * @param $sensor :
	 *			boolean
	 *
	 */
	public function __construct($mapCanvas, $sensor = true) {
		$this->mapCanvas = &$mapCanvas;
		$this->set('apiSensor', $sensor );
		ceHelper::loadJqueryUI();
	}

	/**
	 * @function setWidth
	 * ription Sets the width of the map to be displayed
	 *
	 * @param $width :
	 *			int
	 * @return s nothing
	 */
	function setWidth($width) {
		if($width <= 0) {
			$width = $this->mapWidth;
		}
		$this->mapWidth = $width;
	}

	/**
	 * @function setHeight
	 * ription Sets the height of the map to be displayed
	 *
	 * @param $height :
	 *			int
	 * @return s nothing
	 */
	function setHeight($height) {
		if($height <= 0) {
			$height = $this->mapHeight;
		}
		$this->mapHeight = $height;
	}


	/**
	 * @function setZoomLevel
	 * ription Sets the zoom level of the map (0 is fully zoomed out and 19 is fully zoomed in)
	 *
	 * @param $zoom :
	 *			int (0 - 19)
	 * @return s nothing
	 */
	function setZoomLevel($zoom) {
		if (($zoom <= 0) or ($zoom > 19)) {
			$zoom = $this->zoomLevel;
		}
		$this->zoomLevel = $zoom;
	}

	/**
	 * @function setMapType
	 * ription Sets the type of the map to be displayed, either a (road)map, satellite, hybrid or terrain view; (road)map by default
	 *
	 * @param $mapType :
	 *			string (can be 'ROADMAP', 'SATELLITE', 'HYBRID' or 'TERRAIN')
	 * @return s nothing
	 */
	function setMapType($mapType) {
		switch ($mapType) {
			case 'SATELLITE' :
			case 'HYBRID' :
			case 'TERRAIN' :
				$this->mapType = $mapType;
				break;
			default :
				$this->mapType = 'ROADMAP';
				break;
		}
	}

	/**
	 * @function setInfoWindowBehaviour
	 * ription Sets the behaviour of InfoWindow overlays, either multiple or single windows are displayed
	 *
	 * @param $infoWindowBehaviour :
	 *			string (can be 'MULTIPLE', 'SINGLE' or 'CLOSE_ON_MAPCLICK')
	 * @return s nothing
	 */
	function setInfoWindowBehaviour($infoWindowBehaviour) {
		switch ($infoWindowBehaviour) {
			case 'MULTIPLE' :
			case 'SINGLE' :
			case 'CLOSE_ON_MAPCLICK' :
			case 'SINGLE_CLOSE_ON_MAPCLICK' :
				$this->infoWindowBehaviour = $infoWindowBehaviour;
				break;
			default :
				$this->infoWindowBehaviour = 'MULTIPLE'; // default behaviour of Google Maps V3
				break;
		}
	}

	/**
	 * @function setInfoWindowTrigger
	 * ription Determines if InfoWindow is shown with a click or by mouseover
	 *
	 * @param $infoWindowTrigger :
	 *			string : can be 'CLICK' OR 'ONMOUSEOVER'
	 * @return s nothing
	 */
	function setInfoWindowTrigger($infoWindowTrigger) {
		switch ($infoWindowTrigger) {
			case 'click' :
				$this->infoWindowTrigger = $infoWindowTrigger;
				break;
			default :
				$this->infoWindowTrigger = 'mouseover';
				break;
		}
	}

	/**
	 * @function showDefaultUI
	 * @desc Tells the v3 API wether to show the default UI (its behaviour) or not
	 *
	 * @param $control :
	 *			boolean
	 * @return s nothing
	 */
	function showDefaultUI($control = true) {
		$this->showDefaultUI = $control;
	}

	/**
	 * @function showMapTypeControl
	 * ription Tells the v3 API wether to show the map type control or not
	 *
	 * @param $control :
	 *			boolean
	 * @param $style :
	 *			string (can be 'HORIZONTAL_BAR', 'DROPDOWN_MENU' or 'DEFAULT')
	 * @return s nothing
	 */
	function showMapTypeControl($control = true, $style) {
		$this->showMapTypeControl = $control;

		switch ($style) {
			case 'HORIZONTAL_BAR' :
			case 'DROPDOWN_MENU' :
				$this->mapTypeControlStyle = $style;
				break;
			default :
				$this->mapTypeControlStyle = 'DEFAULT';
				break;
		}
	}

	/**
	 * @function showNavigationControl
	 * ription Tells the v3 API wether to show the navigation control or not
	 *
	 * @param $control :
	 *			boolean
	 * @param $style :
	 *			string (can be 'ANDROID', 'DEFAULT', 'SMALL' or 'ZOOM_PAN')
	 * @return s nothing
	 */
	function showNavigationControl($control = true, $style) {
		$this->showNavigationControl = $control;
		switch ($style) {
			case 'ANDROID' :
			case 'SMALL' :
			case 'ZOOM_PAN' :
				$this->navigationControlStyle = $style;
				break;
			default :
				$this->navigationControlStyle = 'DEFAULT';
				break;
		}
	}

	/**
	 * loadJS
	 * @desc Adds the necessary Javascript for the Google Map API v3 and the jQuery files
	 *
	 * @return s nothing
	 */
	public function loadJS() {
		$doc = JFactory::getDocument ();
		$config = JFactory::getConfig ();
		JHtml::_('jquery.framework');

		$http = 'http' . (ceHelper::httpIsSecure () ? 's://' : '://');
		// Please keep in this order
		$doc->addScript ( '//google.com/jsapi' );
		$doc->addScript ( '//maps.googleapis.com/maps/api/js?sensor=false&amp;language=' . $this->getLanguage () );
		$debug = $config->get ( 'debug' ) OR $config->get ( 'error_reporting' ) == 'development';
		$doc->addScript ( JURI::root(true) . '/components/com_contactenhanced/assets/js/jquery.ui.map/jquery.ui.map.full'.($debug ? '' : '.min').'.js' );
	}

	public function getLanguage() {

		if($this->language == 'joomla') {
			$gmapsLanguage	=	array();
			$gmapsLanguage[]	="ar";	// ARABIC
			$gmapsLanguage[]	="bg";	// BULGARIAN
			$gmapsLanguage[]	="bn";	// BENGALI
			$gmapsLanguage[]	="ca";	// CATALAN
			$gmapsLanguage[]	="cs";	// CZECH
			$gmapsLanguage[]	="da";	// DANISH
			$gmapsLanguage[]	="de";	// GERMAN
			$gmapsLanguage[]	="el";	// GREEK
			$gmapsLanguage[]	="en";	// ENGLISH
			$gmapsLanguage[]	="en-AU";	// ENGLISH (AUSTRALIAN)
			$gmapsLanguage[]	="en-GB";	// ENGLISH (GREAT BRITAIN)
			$gmapsLanguage[]	="es";	// SPANISH
			$gmapsLanguage[]	="eu";	// BASQUE
			$gmapsLanguage[]	="eu";	// BASQUE
			$gmapsLanguage[]	="fa";	// FARSI
			$gmapsLanguage[]	="fi";	// FINNISH
			$gmapsLanguage[]	="fil";	// FILIPINO
			$gmapsLanguage[]	="fr";	// FRENCH
			$gmapsLanguage[]	="gl";	// GALICIAN
			$gmapsLanguage[]	="gu";	// GUJARATI
			$gmapsLanguage[]	="hi";	// HINDI
			$gmapsLanguage[]	="hr";	// CROATIAN
			$gmapsLanguage[]	="hu";	// HUNGARIAN
			$gmapsLanguage[]	="id";	// INDONESIAN
			$gmapsLanguage[]	="it";	// ITALIAN
			$gmapsLanguage[]	="iw";	// HEBREW
			$gmapsLanguage[]	="ja";	// JAPANESE
			$gmapsLanguage[]	="kn";	// KANNADA
			$gmapsLanguage[]	="ko";	// KOREAN
			$gmapsLanguage[]	="lt";	// LITHUANIAN
			$gmapsLanguage[]	="lv";	// LATVIAN
			$gmapsLanguage[]	="ml";	// MALAYALAM
			$gmapsLanguage[]	="mr";	// MARATHI
			$gmapsLanguage[]	="nl";	// DUTCH
			$gmapsLanguage[]	="no";	// NORWEGIAN
			$gmapsLanguage[]	="pl";	// POLISH
			$gmapsLanguage[]	="pt";	// PORTUGUESE
			$gmapsLanguage[]	="pt-BR";	// PORTUGUESE (BRAZIL)
			$gmapsLanguage[]	="pt-PT";	// PORTUGUESE (PORTUGAL)
			$gmapsLanguage[]	="ro";	// ROMANIAN
			$gmapsLanguage[]	="ru";	// RUSSIAN
			$gmapsLanguage[]	="sk";	// SLOVAK
			$gmapsLanguage[]	="sl";	// SLOVENIAN
			$gmapsLanguage[]	="sr";	// SERBIAN
			$gmapsLanguage[]	="sv";	// SWEDISH
			$gmapsLanguage[]	="ta";	// TAMIL
			$gmapsLanguage[]	="te";	// TELUGU
			$gmapsLanguage[]	="th";	// THAI
			$gmapsLanguage[]	="tl";	// TAGALOG
			$gmapsLanguage[]	="tr";	// TURKISH
			$gmapsLanguage[]	="uk";	// UKRAINIAN
			$gmapsLanguage[]	="vi";	// VIETNAMESE
			$gmapsLanguage[]	="zh-CN";	// CHINESE (SIMPLIFIED)
			$gmapsLanguage[]	="zh-TW";	// CHINESE (TRADITIONAL)

			$lang 	=	JFactory::getLanguage();
			$tag	=	$lang->getTag();
			if(in_array($tag,$gmapsLanguage)){
				return $tag;
			}elseif(in_array(substr($tag,0,2),$gmapsLanguage)){
				return substr($tag,0,2);
			}
		}elseif($this->language == 'auto') {
			return '';
		}
		return $this->language;
	}

	/**
	 * @function adjustCenterCoords
	 *
	 * ription adjust map center coordinates by the given lat/lon point
	 *
	 * @param $lng the
	 *			map longitude : string
	 * @param $lat the
	 *			map latitude : string
	 */
	private function adjustCenterCoords($lat, $lng) {
		if ((strlen ( ( string ) $lat ) != 0) and (strlen ( ( string ) $lng ) != 0)) {
			$this->maxLat = ( float ) max ( $lat, $this->maxLat );
			$this->minLat = ( float ) min ( $lat, $this->minLat );
			$this->maxLng = ( float ) max ( $lng, $this->maxLng );
			$this->minLng = ( float ) min ( $lng, $this->minLng );

			$this->centerLng = ( float ) ($this->minLng + $this->maxLng) / 2;
			$this->centerLat = ( float ) ($this->minLat + $this->maxLat) / 2;
		}
	}

	/**
	 * addMarker
	 * @desc Add's a Marker to be displayed on the Google Map using latitude/longitude
	 *
	 * @param 	$lat 		string (latitude)
	 * @param 	$lng 		string (longitude)
	 * @param 	$tooltip 	string (tooltip text)
	 * @param 	$info 		string Message to be displayed in an InfoWindow
	 * @param 	$iconURL 	string  URL to an icon to be displayed instead of the default icon (see for example http://code.google.com/p/google-maps-icons/)
	 * @param 	$clickable 	boolean (true if the marker should be clickable)
	 */
	function addMarker($lat,$lng,$id=null, $info="", $tooltip='', $tags='', $iconURL="", $clickable=true, $dragable = false, $address = '') { //$imageSize = array('width' => 130, 'height' => 130)
		$count = count ( $this->mapMarkers );
		$this->mapMarkers [$count] ['id'] 		= "'".(($id) ? $id : 'm_'.str_replace(array(',','.',' '), "",$lat.$lng))."'";
		$this->mapMarkers [$count] ['lat'] 		= $lat;
		$this->mapMarkers [$count] ['lng'] 		= $lng;
		$this->mapMarkers [$count] ['title'] 	= addslashes($tooltip);
		$this->mapMarkers [$count] ['iconURL'] 	= $iconURL;
		$this->mapMarkers [$count] ['tags'] 	= addslashes($tags);
		$this->mapMarkers [$count] ['clickable']= $clickable;
		$this->mapMarkers [$count] ['dragable']	= $dragable;
		$this->mapMarkers [$count] ['address']	= addslashes(str_replace(array("\n","\r"), '', $address));

		// Makes sure the InfoWindow is not actually a javascript
		if (is_string($info)
				AND (strpos($info, '$') > 0	OR strpos($info, '$') === false	)
		) {

			$info	= "'{$info}'";
		}

		$info = array($info);

		if($this->showCoordinates){
			$info[] = "'{$this->getCoordinatesDiv($lat,$lng)}'";
		}

		if($this->get('useDirections')){
			$info[] = "$('#cemap_directions_container').html()";
		}

		$this->mapMarkers [$count] ['info'] 	= implode('+', $info);

		$this->adjustCenterCoords ( $lat, $lng );
	}

	/**
	 * Get the content of the map InfoWindow
	 * @return	string
	 */
	public function getInfoWindowContentFromContact(&$info, $content = 'address'){
		$html ="";

		if(is_object($info)){
			
			if(!isset($info->params) OR !$info->params instanceof Registry){
				$info->params = JComponentHelper::getParams('com_contactenhanced');
			}
			
			if($content == 'address'){
				$webpage	= '';
				$address	= array();

				$address	= array( 	$info->suburb,
										$info->state,
										$info->postcode,
										$info->country
								);

				foreach ($address as $key => $value) {
					if (is_null($value) OR $value == '') {
						unset($address[$key]);
					}
				}
			$address_str	      = '';
			
			 if (!is_null($info->address) AND $info->address != '' OR count($address)) {
				 $address_str      .= '<p class="address">';
				 if (!is_null($info->address) AND $info->address != '') {
					 $address_str      .= htmlspecialchars($info->address).'<br />';
				 }
				 $address_str      .=implode('<br />',$address);
				 $address_str      .= '</p>';
			 }
			
			
			 if (!is_null($info->telephone) AND $info->telephone != '') {
				 $address_str      .= '<p class="telephone">'.$info->telephone.'</p> ';
			 }
			 if (!is_null($info->webpage) AND $info->webpage != '') {
				 $address_str      .= '<p class="webpage">'.JHTML::_('link'
						,$info->webpage
						,JStringPunycode::urlToUTF8($info->webpage)
						,'target="_blank"').'</p> ';
			}

				if($info->image){
					$attributes	= array('style' => 'float:right');
					$width		= (int)$info->params->get('show_image-resize-width');
					$height		= (int)$info->params->get('show_image-resize-height');

					if($width > 130 	OR $width <= 0){	$width=130;}
					if($height > 130 	OR $height <= 0){	$height=130;}

					if($info->params->get('show_image') == 'resize'){
						$info->params->set('thumbnail_mode', $info->params->get('show_image-resize-mode', 'crop'));
						$info->params->set('thumbnail_mode-resize-use_ratio', $info->params->get('show_image-resize-use_ratio', 0));
						require_once (JPATH_ROOT.'/components/com_contactenhanced/helpers/image.php');

						$html .= ceRenderImage( JText::_('COM_CONTACTENHANCED_IMAGE_DETAILS')
								, $info->image
								, $info->params
								, $width
								, $height
								, null
								, $attributes
						);
					}else{
						$attributes['style']	= "float:right;width:$width".(strpos($width, 'px')? '' : 'px').";";
						//$html .= JHTML::_('image',JURI::base(true).$info->image, JText::_('COM_CONTACTENHANCED_IMAGE_DETAILS'), $attributes);
						$html .= '<img src="'.JURI::base().'/'.$info->image.'"
									alt="'.JText::_('COM_CONTACTENHANCED_IMAGE_DETAILS').'"
									'.trim((is_array($attributes) ? JArrayHelper::toString($attributes) : $attributes)).' />';
					}
				}
				$html .= '<div>'.$this->nl2brStrict($address_str).'</div>';

			}elseif(isset($info->$content)){
				$html .= str_replace(array("\r\n", "\n", "\r"), '', $info->$content);
			}
			if($html){
				$html = '<div class="ce-map-infowindow">'.$html.'</div>';
				$html 	= str_replace("'", "&#8216;", $html);
				$html 	= str_replace(array("\r\n", "\n", "\r"), '', $html);
				$html 	= addslashes($html);
			}

		}
		return $html;
	}

	public function getCoordinatesDiv($lat,$lng) {
		if(count ( $this->mapMarkers ) < 2 AND !$this->editMode){
			$html	= '<div class="ce-map-coordinates">';
				$html	.= '<div class="ce-map-lat">';
					$html	.= '<span class="ce-map-coord-label">'.JText::_('CE_GMAPS_LATITUTE', true).': </span>' ;
					$html	.= '<span class="ce-map-coord-value">'.$lat.'</span>' ;
				$html	.= '</div>';
				$html	.= '<div class="ce-map-lng">';
					$html	.= '<span class="ce-map-coord-label">'.JText::_('CE_GMAPS_LONGITUTE', true).': </span>' ;
					$html	.= '<span class="ce-map-coord-value">'.$lng.'</span>' ;
				$html	.= '</div>';
			$html	.= '</div>';
		}
		return $html;
	}

	public function getDirectionsDiv() {
		$html = '';
		if($this->get('useDirections')){
			$html .= '<div id="cemap_directions_container" style="margin-top:5px"><div id="cemap_directions">';
			$html .= '
				<div class="input-prepend input-group" title="'.JText::_('CE_GMAPS_GET_DIRECTIONS').'" style="width:100%">
					<span class="add-on input-group-addon"><i class="icon-road glyphicon glyphicon-road"> </i></span>
					<input id="locality" name="locality" type="search" 
						class="cemap_address" 
						placeholder="'.JText::_('CE_GMAPS_ENTER_YOUR_ADDRESS').'"
						 />
				</div>';

			$html .= '<div>';
			$html .= '
				<div class="btn-group gmaps_avoid_options" data-toggle="buttons-checkbox"
					'.($this->showAvoidTolls OR $this->showAvoidHighways ? '' : ' style="display:none"').'>'
					.'<button type="button" class="btn btn-default" id="avoid_highways"
							'.($this->showAvoidHighways ? '' : ' style="display:none"').'>'.JText::_('CE_GMAPS_ROUTE_AVOID_HIGHWAYS').'</button>'
					.'<button type="button" class="btn btn-default" id="avoid_tolls"
							'.($this->showAvoidTolls ? '' : ' style="display:none"').'>'.JText::_('CE_GMAPS_ROUTE_AVOID_TOLLS').'</button>'
				.'</div>
			';

			$html .= '
				<div class="btn-group gmaps_travel_mode" data-toggle="buttons-radio"
						'.($this->travelMode == 'show_option' ? '' : ' style="display:none"').'>'
						.'<button type="button" class="btn active btn-driving" 					data-value="DRIVING"	title="'.JText::_('CE_GMAPS_DRIVING_MODE_DRIVING').'"	>
						<img src="//maps.gstatic.com/mapfiles/transparent.png" />
					</button>'
								.'<button type="button" class="btn btn-public-transit"			data-value="TRANSIT"	title="'.JText::_('CE_GMAPS_DRIVING_MODE_PUBLIC_TRANSIT').'"	>
						<img src="//maps.gstatic.com/mapfiles/transparent.png" />
					</button>'
										.'<button type="button" class="btn btn-bicycling"		data-value="BICYCLING"	title="'.JText::_('CE_GMAPS_DRIVING_MODE_BICYCLING').'">
						<img src="//maps.gstatic.com/mapfiles/transparent.png" />
					</button>'
												.'<button type="button" class="btn btn-walking"	data-value="WALKING"	title="'.JText::_('CE_GMAPS_DRIVING_MODE_WALKING').'"	>
						<img src="//maps.gstatic.com/mapfiles/transparent.png" />
					</button>'
				.'</div>
			';

			$html .= '</div>';

			$html .= '</div></div>';
		}

		return $html;
	}

	/**
	 * @function addMarkerByAddress
	 * @desc Add's a Marker to be displayed on the Google Map using latitude/longitude
	 *
	 * @param $lat :
	 *			string (latitude)
	 *			$lng : string (longitude)
	 *			$tooltip : string (tooltip text)
	 *			$info : Message to be displayed in an InfoWindow
	 *			$iconURL : URL to an icon to be displayed instead of the default icon
	 *			(see for example http://code.google.com/p/google-maps-icons/)
	 *			$clickable : boolean (true if the marker should be clickable)
	 */
	function addMarkerByAddress($address, $tooltip = "", $info = "", $iconURL = "", $clickable = true) {
		$geoCoder = new simpleGMapGeocoder ();
		$result = array ();

		if (! is_string ( $address )) {
			die ( "All Addresses must be passed as a string" );
		}

		$result = $geoCoder->getGeoCoords ( $address );

		if($result ['status'] == "OK") {
			$count = count ( $this->mapMarkers );
			$this->mapMarkers [$count] ['lat'] = $result ['lat'];
			$this->mapMarkers [$count] ['lng'] = $result ['lng'];
			$this->mapMarkers [$count] ['tooltip'] = $tooltip;
			$this->mapMarkers [$count] ['info'] = $info;
			$this->mapMarkers [$count] ['iconURL'] = $iconURL;
			$this->mapMarkers [$count] ['clickable'] = $clickable;
			$this->mapMarkers [$count] ['address'] = addslashes(str_replace(array("\n","\r"), '', $address));

			if($this->showCoordinates){
				$this->mapMarkers [$count] ['info'] .= $this->getCoordinatesDiv($result ['lat'],$result ['lng']);
			}

			$this->adjustCenterCoords ( $result ['lat'], $result ['lng'] );
		}
	}

	/**
	 * @function addCircle
	 * ription Add's an circle to be displayed on the Google Map using latitude/longitude and radius
	 *
	 * @param $lat :
	 *			string (latitude)
	 *			$lng : string (longitude)
	 *			$rad : string (radius of circle in meters)
	 *			$info : Message to be displayed in an InfoWindow
	 *			$options : array (options like stroke color etc. for the circle)
	 */
	function addCircle($lat, $lng, $rad, $info = "", $options = array()) {
		$count = count ( $this->mapCircles );
		$this->mapCircles [$count] ['lat'] = $lat;
		$this->mapCircles [$count] ['lng'] = $lng;
		$this->mapCircles [$count] ['rad'] = $rad;
		$this->mapCircles [$count] ['info'] = $info;

		/* set options */
		if (sizeof ( $options ) != 0) {
			$this->mapCircles [$count] ['fillColor'] = $options ['fillColor'];
			$this->mapCircles [$count] ['fillOpacity'] = $options ['fillOpacity'];
			$this->mapCircles [$count] ['strokeColor'] = $options ['strokeColor'];
			$this->mapCircles [$count] ['strokeOpacity'] = $options ['strokeOpacity'];
			$this->mapCircles [$count] ['strokeWeight'] = $options ['strokeWeight'];

			if($options ['clickable'] == "" or $options ['clickable'] == false) {
				$this->mapCircles [$count] ['clickable'] = false;
			}else{
				$this->mapCircles [$count] ['clickable'] = true;
			}
		}
		$this->adjustCenterCoords ( $lat, $lng );
	}

	/**
	 * @function addRectangle
	 * ription Add's a rectangle to be displayed on the Google Map using latitude/longitude for soutwest and northeast corner
	 *
	 * @param $lat1 :
	 *			string (latitude sw corner)
	 *			$lng1 : string (longitude sw corner)
	 *			$lat2 : string (latitude ne corner)
	 *			$lng2 : string (longitude ne corner)
	 *			$info : Message to be displayed in an InfoWindow
	 *			$options : array (options like stroke color etc. for the rectangle)
	 */
	function addRectangle($lat1, $lng1, $lat2, $lng2, $info = "", $options = array()) {
		$count = count ( $this->mapRectangles );
		$this->mapRectangles [$count] ['lat1'] = $lat1;
		$this->mapRectangles [$count] ['lng1'] = $lng1;
		$this->mapRectangles [$count] ['lat2'] = $lat2;
		$this->mapRectangles [$count] ['lng2'] = $lng2;
		$this->mapRectangles [$count] ['info'] = $info;

		/* set options */
		if (sizeof ( $options ) != 0) {
			$this->mapRectangles [$count] ['fillColor'] = $options ['fillColor'];
			$this->mapRectangles [$count] ['fillOpacity'] = $options ['fillOpacity'];
			$this->mapRectangles [$count] ['strokeColor'] = $options ['strokeColor'];
			$this->mapRectangles [$count] ['strokeOpacity'] = $options ['strokeOpacity'];
			$this->mapRectangles [$count] ['strokeWeight'] = $options ['strokeWeight'];

			if($options ['clickable'] == "" or $options ['clickable'] == false) {
				$this->mapRectangles [$count] ['clickable'] = false;
			}else{
				$this->mapRectangles [$count] ['clickable'] = true;
			}
		}
		$this->adjustCenterCoords ( $lat1, $lng1 );
		$this->adjustCenterCoords ( $lat2, $lng2 );
	}

	/**
	 * @function calculateDistance
	 * calculates distance between two locations in given unit (default kilometers)
	 *
	 * @param $lat1 :
	 *			string (latitude location 1)
	 *			$lng1: string (longitude location 1)
	 *			$lat2 : string (latitude location 2)
	 *			$lng2: string (longitude location 2)
	 *			$unit : km (killometers), m (miles), n (nautical miles), i (inch)
	 */
	function calculateDistance($lat1, $lng1, $lat2, $lng2, $unit = "km") {
		$radius = 6371; // mean radius of the earth in kilometers
		$lat1 = ( float ) $lat1;
		$lat2 = ( float ) $lat2;
		$lng1 = ( float ) $lng1;
		$lng2 = ( float ) $lng2;

		// calculation of distance in km using Great Circle Distance Formula
		$dist = $radius * acos ( sin ( deg2rad ( $lat1 ) ) * sin ( deg2rad ( $lat2 ) ) + cos ( deg2rad ( $lat1 ) ) * cos ( deg2rad ( $lat2 ) ) * cos ( deg2rad ( $lng2 ) - deg2rad ( $lng1 ) ) );

		switch (strtolower ( $unit )) {
			case 'm' : // miles
				$dist = $dist / 1.609;
				break;
			case 'n' : // nautical miles
				$dist = $dist / 1.852;
				break;
			case 'i' : // inch
				$dist = $dist * 39370;
				break;
		}

		return $dist;
	}

	/**
	 * @function showMap
	 * ription Displays the Google Map on the page
	 */
	function showMap($zoomToBounds = true) {
		
		$this->showNavigationControl(($this->get('navigationControl') ? true : false), $this->get('navigationControl'));
		$html 	= "\n<!-- DIV container for the map -->\n";
		$style = 'style="margin:0;'.($this->mapHeight > 50 ? "height:{$this->mapHeight}px;" : '').'"';
		// create div for the map canvas
		$html 	.= '
				<div class="cemap_container">
';
		if($this->get('useDirections') == 'before_map'){
			$html 	.= '<div id="gmaps_directions_panel"></div>';
		}
		$html 	.= '<div id="'.$this->mapCanvas.'"
						'.$style.'
						class="cemap_canvas">
					</div>';
		if($this->get('useDirections') == '1' OR $this->get('useDirections') == 'after_map'){
			$html 	.= '<div id="gmaps_directions_panel"></div>';
		}
		$html 	.= '<br style="clear:both" />
				</div>
				';
		$markers = $this->getMarkersScript();
		$autoCloseInfowindow 	= "google.maps.event.addListener(map,'click',function(){jQuery('#{$this->mapCanvas}').gmap('closeInfoWindow');});";

		$script 	= '
jQuery(document).ready(function($){'."
setTimeout(function(){
	var mapOptions = {
		scrollwheel:		{$this->get('scrollwheelZoom', 1)},
		mapTypeId:			google.maps.MapTypeId.{$this->get('mapType')},
		draggable:			{$this->get('mapDraggable')},
		mapTypeControl:		{$this->get('mapTypeControl')},
		mapTypeControlOptions:{style: google.maps.MapTypeControlStyle.{$this->mapTypeControlStyle}},
		navigationControlOptions:{style:google.maps.NavigationControlStyle.".$this->navigationControlStyle." },
		navigationControl:	".($this->get('navigationControl')	? 'true' : 'false').",
		panControl: 		".($this->get('panControl')			? 'true' : 'false').",
		zoomControl: 		".($this->get('zoomControl')		? 'true' : 'false').",
		zoomControlOptions: {
			style: google.maps.ZoomControlStyle.{$this->get('zoomControl','DEFAULT')}, 
			position: google.maps.ControlPosition.LEFT_CENTER
		},
		streetViewControl:	".($this->get('streetViewControl')	? 'true' : 'false').",
		scaleControl:		{$this->get('scaleControl')},
		".(trim($this->get('mapStyles')) ? 'styles:	'.$this->get('mapStyles', '{}') : '')."
	}
	var currentMarker = null;
	jQuery('#{$this->mapCanvas}').gmap(mapOptions).bind('init', function(event, map) {

		";
		$script 	.= $markers;
		
		$script 	.= $autoCloseInfowindow;
		if(sizeof ( $this->mapMarkers ) < 2){
			$script 	.= "\n$('#{$this->mapCanvas}').gmap('option', 'zoom', {$this->get('zoomLevel',10)});\n";
		}
		$script 	.= "
	}); // End Bind";

		// in case the user is using Display = tabs.
		// Ask the user to display the infoWindow on Marker click if display = tabs.
		$script 	.= "
		$('dt.form-map span h3 a, .accordion-toggle, ul.nav-tabs a').mouseout(function(e){setTimeout(function(){ $('#{$this->mapCanvas}').gmap('refresh');},500);});
		$('#{$this->mapCanvas}').gmap('refresh');
}, 350); // End settimeout
}); // End jQuery Ready
";

		$doc = JFactory::getDocument();
		$doc->addScriptDeclaration($script);

		$this->loadJS();

		if($this->get('useDirections')){
			$html .= $this->getDirectionsDiv();
		}

		return $html;
	}

	
	private function getMarkersScript(){
		$infoWindowTrigger 	= $this->get('infoWindowTrigger') == 'mouseover' ? 'mouseover' : 'click';
		
		// AutoLoad Info Window if set to alwaysOn and there is only one marker
		$autoLoadinfoWindow = '';
		if ( ($this->get('infoWindowTrigger') == 'alwaysOn' AND sizeof($this->mapMarkers) == 1)) {
			$autoLoadinfoWindow  = ", function(map,marker) {
					setTimeout(function (){ $(marker).triggerEvent('click');},2500);
				}";
		}
		
		// create JS to display the map
		/*
		 * Run through the mapMarkers array to display markers on the map
		*/
		$markers ='';
		for($count = 0; $count < sizeof ( $this->mapMarkers ); $count ++) {
		$marker = &$this->mapMarkers[$count];
		$marker['tags'] = str_replace("'", '', $marker['tags']);
		
		$markers 	.= "
		$('#{$this->mapCanvas}').gmap('addMarker',
			{'id': {$marker['id']}
				, 'tags': 	'{$marker['tags']}'
				, 'position': new google.maps.LatLng({$marker['lat']}, {$marker['lng']})
				, 'title': '{$marker['title']}'
				, 'bounds':true
				, 'address': '{$marker['address']}'
				".($marker['iconURL'] ? ", 'icon': '".$marker['iconURL']."'" : '')."
			}{$autoLoadinfoWindow})";
		
		$linkToGoogleMaps = '';
		if($this->get('linkToGoogleMaps')){
		$linkToGoogleMaps = "
var mapFrom = new String(ui.item.position).replace(' ', '+');
var mapTo 	= new String(fromMarker.position).replace(' ', '+');
jQuery('<a/>',{
    href: 'https://maps.google.com/?saddr='+mapFrom+'&daddr='+mapTo,
    rel: 'external',
    target: '_blank',
    text: '".JText::_('CE_GMAPS_OPEN_IN_TO_GOOGLE_MAPS',true)."'
}).appendTo('#gmaps_directions_panel');
";
		}
		
		$closeDirectionsButton = "$('<button/>',{
    class:	'close',
	style:	'float:right',
	title:	'".JText::_('JLIB_HTML_BEHAVIOR_CLOSE',true)."'
}).appendTo('#gmaps_directions_panel').html('&times;').on('click', function(){ $('#{$this->mapCanvas}').gmap('removeDirections'); $('#gmaps_directions_panel').html('');});";
		
		
		if($marker['info']){
			$markers .=".".$infoWindowTrigger."(function() {
$('#{$this->mapCanvas}').gmap('openInfoWindow', { 'content': {$marker['info']} }, this);
	fromMarker = this;
	".($this->get('useDirections') ? "setTimeout(function(){
		var autoCompleteResult = $('#{$this->mapCanvas}').gmap('autocomplete','locality',fromMarker, function(ui,fromMarker){
			;
			var detination = fromMarker.address ? fromMarker.address : fromMarker.position;
			$('#{$this->mapCanvas}').gmap('displayDirections'
				,{'origin':ui.item.position
					,'destination':	detination
					,'travelMode':	$('.gmaps_travel_mode .active').data('value')
					, provideRouteAlternatives: true
					, avoidHighways:$('#avoid_highways').hasClass('active')
					, avoidTolls:	$('#avoid_tolls').hasClass('active')
				}
			,{	'panel':document.getElementById('gmaps_directions_panel')
				, draggable:	true
			}
			,function(result,status){
				if ( status === 'OK' ) {
					{$closeDirectionsButton}
					{$linkToGoogleMaps}
				}else{ // IF Failed, then try to get directions based on position instead of address, it's less accurate, but always works
					$('#{$this->mapCanvas}').gmap('displayDirections'
						,{'origin':ui.item.position
							,'destination':	fromMarker.position
							,'travelMode':	$('.gmaps_travel_mode .active').data('value')
							, provideRouteAlternatives: true
							, avoidHighways:$('#avoid_highways').hasClass('active')
							, avoidTolls:	$('#avoid_tolls').hasClass('active')
						}
					,{	'panel':document.getElementById('gmaps_directions_panel')
						, draggable:	true
					}
					,function(result,status){
						if ( status === 'OK' ) {
							{$closeDirectionsButton}
							{$linkToGoogleMaps}
						}
					});
				
				}
			});
		}); // end autocomplete
	},1500);" : '')."
})";
	}
	$markers .="; // End Marker
	";
		}
		return $markers;
	}
	private function nl2brStrict($text, $replacement = '<br />')
	{
		return preg_replace("((\r\n)+)", trim($replacement), $text);
	}
}
// End Of Class

?>
